<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
requireLogin();

// Log session information
error_log('Session user ID: ' . ($_SESSION['user_id'] ?? 'Not set'));
error_log('Session data: ' . json_encode($_SESSION));

header('Content-Type: application/json');

// Enable error logging
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

try {
    // Log received data for debugging
    error_log('Quotation creation request received: ' . json_encode($_POST));
    error_log('Request method: ' . $_SERVER['REQUEST_METHOD']);
    error_log('Content type: ' . ($_SERVER['CONTENT_TYPE'] ?? 'Not set'));
    
    // Get form data
    $customer_id = !empty($_POST['customer_id']) ? intval($_POST['customer_id']) : null;
    $valid_until = $_POST['valid_until'] ?? '';
    $notes = $_POST['notes'] ?? '';
    $product_ids = $_POST['product_id'] ?? [];
    $quantities = $_POST['quantity'] ?? [];
    $unit_prices = $_POST['unit_price'] ?? [];
    
    // Log processed data
    error_log('Processed data - customer_id: ' . $customer_id . ', valid_until: ' . $valid_until . ', product_ids count: ' . count($product_ids));
    error_log('Product IDs: ' . json_encode($product_ids));
    error_log('Quantities: ' . json_encode($quantities));
    error_log('Unit Prices: ' . json_encode($unit_prices));
    
    // Validate required fields
    error_log('Validating required fields...');
    error_log('Valid until: "' . $valid_until . '" (empty: ' . (empty($valid_until) ? 'true' : 'false') . ')');
    error_log('Product IDs empty: ' . (empty($product_ids) ? 'true' : 'false'));
    error_log('Quantities empty: ' . (empty($quantities) ? 'true' : 'false'));
    error_log('Unit prices empty: ' . (empty($unit_prices) ? 'true' : 'false'));
    
    if (empty($valid_until)) {
        throw new Exception('Valid until date is required');
    }
    
    if (empty($product_ids) || empty($quantities) || empty($unit_prices)) {
        throw new Exception('At least one product item is required');
    }
    
    // Handle empty customer_id (Walk-in Customer)
    if (empty($customer_id)) {
        $customer_id = null; // Allow null for walk-in customers
    }
    
    // Validate arrays have same length
    if (count($product_ids) !== count($quantities) || count($product_ids) !== count($unit_prices)) {
        throw new Exception('Invalid product data');
    }
    
    // Validate individual items
    $items = [];
    $total_amount = 0;
    $valid_items = 0;
    
    for ($i = 0; $i < count($product_ids); $i++) {
        $product_id = intval($product_ids[$i]);
        $quantity = floatval($quantities[$i]);
        $unit_price = floatval($unit_prices[$i]);
        
        // Skip empty rows
        if ($product_id <= 0 || $quantity <= 0 || $unit_price <= 0) {
            continue;
        }
        
        $total_price = $quantity * $unit_price;
        $total_amount += $total_price;
        $valid_items++;
        
        $items[] = [
            'product_id' => $product_id,
            'quantity' => $quantity,
            'unit_price' => $unit_price,
            'total_price' => $total_price
        ];
    }
    
    // Ensure at least one valid item
    if ($valid_items === 0) {
        throw new Exception('At least one product with valid quantity and price is required');
    }
    
    // Start transaction with retry mechanism for quote number generation
    $maxRetries = 3;
    $retryCount = 0;
    $quotation_id = null;
    $quote_number = null;
    
    error_log('Starting quotation creation transaction...');
    
    do {
        try {
            error_log("Attempt " . ($retryCount + 1) . " of $maxRetries");
    $pdo->beginTransaction();
            error_log('Transaction started successfully');
            
            // Generate quote number manually to avoid race conditions
            $currentYear = date('Y');
            error_log("Generating quote number for year: $currentYear");
            
            $stmt = $pdo->prepare("
                SELECT COALESCE(MAX(CAST(SUBSTRING(quote_number, 10) AS UNSIGNED)), 0) + 1 as next_number
                FROM quotations 
                WHERE quote_number LIKE ?
            ");
            $stmt->execute(["QUO-{$currentYear}-%"]);
            $nextNumber = $stmt->fetchColumn();
            $quote_number = "QUO-{$currentYear}-" . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
            
            error_log("Generated quote number: $quote_number (next number: $nextNumber)");
            
            // Insert quotation with explicit quote number
            error_log("Inserting quotation with data: quote_number=$quote_number, customer_id=$customer_id, total_amount=$total_amount, notes='$notes', valid_until=$valid_until, created_by=" . $_SESSION['user_id']);
            
        $stmt = $pdo->prepare("
                INSERT INTO quotations (quote_number, customer_id, total_amount, status, notes, valid_until, created_by)
                VALUES (?, ?, ?, 'active', ?, ?, ?)
        ");
        $stmt->execute([
                $quote_number,
            $customer_id,
            $total_amount,
            $notes,
            $valid_until,
            $_SESSION['user_id']
        ]);
        
        $quotation_id = $pdo->lastInsertId();
            error_log("Quotation inserted successfully with ID: $quotation_id");
        
        // Insert quotation items
            error_log("Inserting " . count($items) . " quotation items");
            
        $stmt = $pdo->prepare("
            INSERT INTO quotation_items (quotation_id, product_id, quantity, unit_price, total_price)
            VALUES (?, ?, ?, ?, ?)
        ");
        
            foreach ($items as $index => $item) {
                error_log("Inserting item $index: quotation_id=$quotation_id, product_id=" . $item['product_id'] . ", quantity=" . $item['quantity'] . ", unit_price=" . $item['unit_price'] . ", total_price=" . $item['total_price']);
                
            $stmt->execute([
                $quotation_id,
                $item['product_id'],
                $item['quantity'],
                $item['unit_price'],
                $item['total_price']
            ]);
        }
            
            error_log("All quotation items inserted successfully");
        
        // Commit transaction
            error_log("Committing transaction...");
        $pdo->commit();
            error_log("Transaction committed successfully");
            break; // Success, exit retry loop
            
        } catch (PDOException $e) {
            error_log("PDO Exception caught: " . $e->getMessage() . " (Code: " . $e->getCode() . ")");
            $pdo->rollBack();
            error_log("Transaction rolled back");
            
            // Check if it's a duplicate key error
            if ($e->getCode() == 23000 && strpos($e->getMessage(), 'quote_number') !== false) {
                error_log("Duplicate quote number error detected, retrying...");
                $retryCount++;
                if ($retryCount >= $maxRetries) {
                    throw new Exception('Failed to generate unique quote number after ' . $maxRetries . ' attempts. Please try again.');
                }
                // Small delay before retry
                usleep(100000); // 0.1 second
                continue;
            } else {
                // Not a duplicate key error, re-throw
                error_log("Non-duplicate PDO error, re-throwing: " . $e->getMessage());
                throw $e;
            }
        }
    } while ($retryCount < $maxRetries);
        
        echo json_encode([
            'success' => true,
            'message' => 'Quotation created successfully',
            'quote_number' => $quote_number,
            'quotation_id' => $quotation_id,
            'total_amount' => $total_amount
        ]);
        
    } catch (Exception $e) {
    error_log('Quotation creation error: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
    error_log('Quotation creation stack trace: ' . $e->getTraceAsString());
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'debug_info' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ]
    ]);
} catch (Error $e) {
    error_log('Quotation creation fatal error: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
    error_log('Quotation creation fatal error stack trace: ' . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Fatal error: ' . $e->getMessage(),
        'debug_info' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ]
    ]);
}
?> 